<?php
/**
 * Response Utility Class
 * 
 * Standardized JSON response format for API
 */

class Response {
    /**
     * Send success response
     * 
     * @param mixed $data
     * @param string $message
     * @param int $statusCode
     */
    public static function success($data = null, $message = "Success", $statusCode = 200) {
        http_response_code($statusCode);
        echo json_encode([
            'success' => true,
            'message' => $message,
            'data' => $data
        ], JSON_UNESCAPED_UNICODE);
        exit();
    }

    /**
     * Send error response
     * 
     * @param string $message
     * @param int $statusCode
     * @param mixed $errors
     */
    public static function error($message = "Error", $statusCode = 400, $errors = null) {
        http_response_code($statusCode);
        $response = [
            'success' => false,
            'message' => $message
        ];
        
        if ($errors !== null) {
            $response['errors'] = $errors;
        }
        
        echo json_encode($response, JSON_UNESCAPED_UNICODE);
        exit();
    }

    /**
     * Send unauthorized response
     * 
     * @param string $message
     */
    public static function unauthorized($message = "Unauthorized") {
        self::error($message, 401);
    }

    /**
     * Send forbidden response
     * 
     * @param string $message
     */
    public static function forbidden($message = "Forbidden") {
        self::error($message, 403);
    }

    /**
     * Send not found response
     * 
     * @param string $message
     */
    public static function notFound($message = "Resource not found") {
        self::error($message, 404);
    }

    /**
     * Send validation error response
     * 
     * @param array $errors
     */
    public static function validationError($errors) {
        self::error("Validation failed", 422, $errors);
    }

    /**
     * Send server error response
     * 
     * @param string $message
     */
    public static function serverError($message = "Internal server error") {
        self::error($message, 500);
    }
}
